import { Component, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { ActivatedRoute } from '@angular/router';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { Subject, concatMap, takeUntil } from 'rxjs';
import { environment } from '../../../environments/environment';
import { AbstractBasicComponent } from '../../core/abstract/abstract-component';
import { Goods } from '../../core/model/goods.model';
import { Order } from '../../core/model/order.model';
import { PagedResponseModel } from '../../core/model/paged-response.model';
import { GoodsService } from '../../core/service/goods.service';
import { OrderLineService } from '../../core/service/order-line.service';
import { OrderService } from '../../core/service/order.service';
import { AddProductModalComponent } from './components/add-product-modal/add-product-modal.component';

const DEV = !environment.production;

@Component({
  selector: 'app-order-page',
  templateUrl: './order-page.component.html',
  styleUrl: './order-page.component.scss',
})
export class OrderPageComponent
  extends AbstractBasicComponent
  implements OnInit
{
  form: FormGroup;
  orderSubj$ = new Subject<Order>();
  order!: Order;

  private id!: number;

  constructor(
    private orderService: OrderService,
    private orderLineService: OrderLineService,
    private goodsService: GoodsService,
    private modalService: NgbModal,
    fb: FormBuilder,
    route: ActivatedRoute,
  ) {
    super();

    this.form = fb.group({
      client: ['', Validators.required],
      date: ['', Validators.required],
      address: ['', Validators.required],
    });

    route.params
      .pipe(takeUntil(this.destroy$))
      .subscribe((params) => (this.id = +params['id']));
  }

  public get isNew(): boolean {
    return this.id === null || isNaN(this.id);
  }

  ngOnInit(): void {
    if (!this.isNew) {
      this.getData(this.id);
    }
  }

  deleteLine(id: number) {
    this.orderLineService.delete(id).subscribe({
      next: (res) => {
        this.getData(this.order.id);
      },
      error: (err) => {
        if (DEV) {
          console.dir(`Error: ${err}`);
        }
      },
    });
  }

  getData(id: number) {
    this.orderService.getOrder(id).subscribe((order) => {
      this.order = order;
      this.orderSubj$.next(order);
    });
  }

  openAddProductModal(): void {
    const orderModalRef = this.modalService.open(AddProductModalComponent, {
      size: 'xl',
    });
    this.goodsService
      .get()
      .subscribe((resp) => (orderModalRef.componentInstance.goods = resp));
    orderModalRef.componentInstance.order = this.order;
    orderModalRef.componentInstance.pageNumber
      .pipe(concatMap((page: number) => this.goodsService.get(page)))
      .subscribe((resp: PagedResponseModel<Goods>) => {
        orderModalRef.componentInstance.goods = resp;
      });

    orderModalRef.componentInstance.orderLineChange
      .pipe(
        concatMap((v: UpdateCount) => {
          const orderLineId = this.order.orderLines.find(
            (ol) => ol.goods.id === v.goods_id,
          )?.id;
          if (orderLineId) {
            return this.orderLineService.put({
              id: orderLineId,
              order_id: this.order.id,
              goods_id: v.goods_id,
              count: v.count,
            });
          }
          return this.orderLineService.post({
            order_id: this.order.id,
            goods_id: v.goods_id,
            count: v.count,
          });
        }),
      )
      .subscribe((resp: any) => {
        console.dir(resp);
      });
  }

  previousState(): void {
    window.history.back();
  }
}

export interface UpdateCount {
  goods_id: number;
  count: number;
}
