package com.example.testj.service;

import com.example.testj.domain.Order;
import com.example.testj.repository.OrderRepository;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.impl.OrderServiceImpl;
import com.example.testj.service.mapper.OrderMapper;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mapstruct.factory.Mappers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PagedModel;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class OrderServiceTest {
    @InjectMocks
    private OrderServiceImpl orderService;

    @Mock
    private OrderRepository orderRepository;

    @Spy
    private OrderMapper orderMapper = Mappers.getMapper(OrderMapper.class);

    private static List<Order> ordersList = new ArrayList<>();

    @BeforeAll
    static void init() {
        Order order = new Order();
        order.setId(1L);
        order.setClient("client 1");
        order.setAddress("address 1");
        order.setDate(LocalDate.of(2020, 1, 1));

        Order order1 = new Order();
        order1.setId(2L);
        order1.setClient("client 2");
        order1.setAddress("address 2");
        order1.setDate(LocalDate.of(2020, 2, 20));

        ordersList = Stream.of(
                order,
                order1
        ).toList();
    }

    @Test
    void testCreateOrUpdateOrder() {
        // given
        Order order = new Order(null, "client new", LocalDate.of(2021, 1, 1), "address new", null);

        Order result = new Order(
                ordersList.getLast().getId() + 1,
                order.getClient(),
                order.getDate(),
                order.getAddress(),
                null
        );
        // when
        when(orderRepository.save(order)).thenReturn(result);
        OrderDto created = orderService.createOrUpdateOrder(order);
        // then
        assertEquals(result.getId(), created.getId());
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    void testSaveOrderWithEmptyClient() {
        // given
        Order order = new Order(null, "", LocalDate.of(2020, 1, 1), "address new", null);
        // when
        when(orderRepository.save(order)).thenThrow(IllegalArgumentException.class);
        assertThrows(IllegalArgumentException.class, () -> orderService.createOrUpdateOrder(order));
        // then
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    void testSaveOrderWithNullName() {
        // given
        Order order = new Order(null, null,  LocalDate.of(2020, 1, 1), "address", null);
        // when
        when(orderRepository.save(order)).thenThrow(IllegalArgumentException.class);
        assertThrows(IllegalArgumentException.class, () -> orderService.createOrUpdateOrder(order));
        // then
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    void testGetAllOrdersPaged() {
        // when
        when(orderRepository.findAll(any(Pageable.class))).thenReturn(new PageImpl<>(ordersList));
        PagedModel<OrderDto> ordersDtoPagedModel = orderService.getPagedOrders(1,10, "id", "ASC");
        // then
        assertNotNull(ordersDtoPagedModel);
        assertEquals(ordersDtoPagedModel.getContent().size(), 2);
        verify(orderRepository, times(1)).findAll(any(Pageable.class));
    }

    @Test
    void testGetOrder() {
        // given
        Long id = 1L;
        // when
        when(orderRepository.findById(id)).thenReturn(Optional.of(ordersList.getFirst()));
        OrderDto order = orderService.getOrder(id);
        // then
        assertNotNull(order);
        assertEquals(id, order.getId());
        verify(orderRepository, times(1)).findById(any());
    }

    @Test
    void testUpdateOrder() {
        // given
        Long id = 1L;
        Order order = ordersList.getFirst();
        order.setClient("client new");
        order.setAddress("address new");
        order.setDate(LocalDate.of(2020, 2, 2));
        // when
        when(orderRepository.save(order)).thenReturn(order);
        OrderDto updated = orderService.createOrUpdateOrder(order);
        // then
        assertNotNull(updated);
        assertEquals(id, updated.getId());
        assertEquals(ordersList.getFirst().getClient(), updated.getClient());
        assertEquals(ordersList.getFirst().getAddress(), updated.getAddress());
        verify(orderRepository, only()).save(any());
    }

    @Test
    void testPatchOrder() throws JsonProcessingException {
        // given
        Long id = 1L;
        Order order = ordersList.getFirst();
        JsonNode patch = new ObjectMapper().readTree("{\"client\": \"client patch\"}");
        Order patchedOrder = new ObjectMapper().updateValue(order, patch);
        // when
        when(orderRepository.findById(id)).thenReturn(Optional.of(order));
        when(orderRepository.save(patchedOrder)).thenReturn(patchedOrder);
        OrderDto patched = orderService.patchOrder(id, patch);
        // then
        assertNotNull(patched);
        assertEquals(id, patched.getId());
        assertEquals(ordersList.getFirst().getClient(), patched.getClient());
        assertEquals(ordersList.getFirst().getAddress(), patched.getAddress());
        verify(orderRepository, times(1)).save(any());
    }

    @Test
    void testDeleteOrder() {
        // given
        Long id = 2L;
        // when
        when(orderRepository.findById(id)).thenReturn(Optional.of(ordersList.getFirst()));
        doNothing().when(orderRepository).deleteById(id);
        orderService.deleteOrder(id);
        // then
        verify(orderRepository, times(1)).deleteById(id);
    }
}
