package com.example.testj.web.rest;

import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.OrderLineService;
import com.example.testj.service.dto.GoodsDto;
import com.example.testj.service.dto.OrderLineDto;
import com.example.testj.service.dto.OrderLineRequestDto;
import com.example.testj.service.dto.OrderLineUpdateRequestDto;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
public class OrderLineControllerTest {
    @Autowired
    private MockMvc mvc;

    @MockBean
    private OrderLineService orderLineService;

    @Test
    @DisplayName("GET /api/order-line/:id [200 OK]")
    void testGetOrderLine() throws Exception {
        Long id = 1L;
        OrderLineDto orderLineDto = new OrderLineDto(id, new GoodsDto(1L, "product 1", 100.00), 10);
        when(orderLineService.getOrderLine(id)).thenReturn(orderLineDto);
        mvc.perform(get("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpectAll(
                        jsonPath("$.id").value(id),
                        jsonPath("$.count").value(orderLineDto.getCount()),
                        jsonPath("$.goods.id").value(orderLineDto.getGoods().getId()));
    }

    @Test
    @DisplayName("GET /api/order-line/:id [404 Not Found]")
    void testGetOrderLineNotFound() throws Exception {
        Long id = 999L;
        when(orderLineService.getOrderLine(id)).thenThrow(new ResourceNotFoundException());
        mvc.perform(get("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isNotFound());
    }

    @Test
    @DisplayName("POST /api/order-line [201 Created]")
    void testCreateOrderLine() throws Exception {
        Long nextId = 1L;
        OrderLineDto orderLineDto = new OrderLineDto(
                nextId,
                new GoodsDto(1L, "product 1", 100.00),
               10);
        OrderLineRequestDto requestDto = new OrderLineRequestDto(1L, 1L, orderLineDto.getCount());
        String body = new ObjectMapper().writeValueAsString(requestDto);

        when(orderLineService.createOrderLine(any(OrderLineRequestDto.class))).thenReturn(orderLineDto);

        mvc.perform(post("/api/order-line").contentType(MediaType.APPLICATION_JSON).content(body))
                .andExpect(status().isCreated())
                .andExpectAll(
                        jsonPath("$.id").value(orderLineDto.getId()),
                        jsonPath("$.count").value(orderLineDto.getCount()));
    }

    @Test
    @DisplayName("PUT /api/order-line/:id [200 OK]")
    void testUpdateOrderLine() throws Exception {
        Long id = 1L;
        OrderLineDto orderLineDto = new OrderLineDto(
                id,
                new GoodsDto(1L, "product 1", 100.00),
                10);
        OrderLineUpdateRequestDto request = new OrderLineUpdateRequestDto(orderLineDto.getCount());

        String body = new ObjectMapper().writeValueAsString(request);
        when(orderLineService.getOrderLine(id)).thenReturn(orderLineDto);
        when(orderLineService.updateOrderLineCount(id, orderLineDto.getCount())).thenReturn(orderLineDto);

        mvc.perform(put("/api/order-line/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(body))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.count").value(orderLineDto.getCount()));
    }

    @Test
    @DisplayName("DELETE /api/order-line/:id [200 OK]")
    void testDeleteOrderLine() throws Exception {
        Long id = 1L;
        doNothing().when(orderLineService).deleteOrderLine(id);
        mvc.perform(delete("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk());
    }
}
