package com.example.testj.web.rest;

import org.springframework.data.web.PagedModel;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.example.testj.domain.Order;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.OrderService;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.utils.MappingUtils;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;

import jakarta.validation.Valid;

@RestController
@RequestMapping("/api/orders")
@CrossOrigin(origins = "http://localhost:4200")
public class OrderController {
    private final OrderService orderService;
    private final MappingUtils mappingUtils;

    public OrderController(OrderService orderService, MappingUtils mappingUtils) {
        this.orderService = orderService;
        this.mappingUtils = mappingUtils;
    }

    @GetMapping
    public PagedModel<OrderDto> getOrders(
            @RequestHeader(name = "X-Page-Current", defaultValue = "0") Integer page,
            @RequestHeader(name = "X-Page-Size", defaultValue = "10") Integer size,
            @RequestHeader(name = "X-Sort-By", defaultValue = "id") String sortBy,
            @RequestHeader(name = "X-Sort-Direction", defaultValue = "ASC") String sortDir) {
        return orderService.getOrders(page, size, sortBy, sortDir);
    }

    @GetMapping("/{id}")
    public OrderDto getOrder(@PathVariable Long id) {
        return orderService.getOrder(id);
    }

    @PostMapping
    @ResponseStatus(HttpStatus.CREATED)
    public OrderDto createOrder(@Valid @RequestBody Order order) {
        return orderService.createOrUpdateOrder(order);
    }

    @PutMapping("/{id}")
    @ResponseStatus(HttpStatus.OK)
    public OrderDto updateOrder(@PathVariable Long id, @RequestBody OrderDto order) {

        OrderDto updatedOrder = orderService.getOrder(id);
        updatedOrder.setClient(order.getClient());
        updatedOrder.setAddress(order.getAddress());
        updatedOrder.setDate(order.getDate());

        return orderService.createOrUpdateOrder(mappingUtils.toOrder(updatedOrder));
    }

    @PatchMapping("/{id}")
    public OrderDto patchOrder(@PathVariable Long id, @RequestBody JsonNode patch) throws JsonMappingException {
        return orderService.patchOrder(id, patch);
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<HttpStatus> deleteOrder(@PathVariable Long id) throws ResourceNotFoundException {
        try {
            orderService.deleteOrder(id);
            return new ResponseEntity<>(HttpStatus.OK);
        } catch (ResourceNotFoundException e) {
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
    }
}
