package com.example.testj.web.rest;

import com.example.testj.domain.Order;
import com.example.testj.domain.RestErrorInfo;
import com.example.testj.dto.OrderDTO;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.repository.OrderRepository;
import com.example.testj.service.OrderService;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import jakarta.validation.Valid;
import org.springframework.data.web.PagedModel;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Optional;

@RestController
@RequestMapping("/api")
@CrossOrigin(origins = "http://localhost:4200")
public class OrderController {
    private final OrderRepository orderRepository;
    private final OrderService orderService;

    public OrderController(OrderRepository orderRepository, OrderService orderService) {
        this.orderRepository = orderRepository;
        this.orderService = orderService;
    }

    @GetMapping("/orders")
    public PagedModel<Order> getOrders(@RequestHeader(name = "X-Current-Page", defaultValue = "0") Integer page,
                                       @RequestHeader(name = "X-Page-Size", defaultValue = "10") Integer size,
                                       @RequestHeader(name = "X-Sort-By", defaultValue = "id") String sortBy,
                                       @RequestHeader(name = "X-Sort-Direction", defaultValue = "ASC") String sortDir) {
        return orderService.getOrders(page, size, sortBy, sortDir);
    }

    @GetMapping("/orders/{id}")
    public Order getOrder(@PathVariable Long id) {
        return this.orderService.getOrder(id);
    }

    @PostMapping("/orders")
    public Order createOrder(@Valid @RequestBody Order order) {
        return orderService.saveOrUpdateOrder(order);
    }

    @PutMapping("/orders/{id}")
    public Order updateOrder(@PathVariable Long id, @RequestBody Order order) {
            Order updatedOrder = orderService.getOrder(id);
            updatedOrder.setClient(order.getClient());
            updatedOrder.setAddress(order.getAddress());
            updatedOrder.setDate(order.getDate());
            return orderService.saveOrUpdateOrder(updatedOrder);
    }

    @PatchMapping("/orders/{id}")
    public Order patchOrder(@PathVariable Long id, @RequestBody JsonNode patch) throws JsonMappingException {
        return orderService.patchOrder(id, patch);
    }

    @DeleteMapping("/orders/{id}")
    public ResponseEntity<HttpStatus> deleteOrder(@PathVariable Long id) throws ResourceNotFoundException {
        try {
            orderService.deleteOrder(id);
            return new ResponseEntity<>(HttpStatus.OK);
        } catch (ResourceNotFoundException e) {
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }

    }
}
