package com.example.testj.web.rest;

import org.springframework.data.web.PagedModel;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.GoodsService;
import com.example.testj.service.dto.GoodsDto;
import com.example.testj.utils.MappingUtils;

@RestController
@RequestMapping("/api/goods")
@CrossOrigin(origins = "http://localhost:4200")
public class GoodsController {
    private final GoodsService goodsService;
    private final MappingUtils mappingUtils;

    public GoodsController(GoodsService goodsService, MappingUtils mappingUtils) {
        this.goodsService = goodsService;
        this.mappingUtils = mappingUtils;
    }

    @GetMapping
    public PagedModel<GoodsDto> goods(
            @RequestHeader(name = "X-Page-Current", defaultValue = "0") Integer page,
            @RequestHeader(name = "X-Page-Size", defaultValue = "10") Integer size,
            @RequestHeader(name = "X-Sort-By", defaultValue = "id") String sortBy,
            @RequestHeader(name = "X-Sort-Direction", defaultValue = "ASC") String sortDir) {
        return goodsService.getAllGoods(page, size, sortBy, sortDir);
    }

    @GetMapping("/{id}")
    public GoodsDto goods(@PathVariable Long id) {
        return goodsService.getGoods(id);
    }

    @PostMapping
    @ResponseStatus(HttpStatus.CREATED)
    public GoodsDto createGoods(@RequestBody GoodsDto goods) {
        return this.goodsService.createOrUpdate(mappingUtils.toGoods(goods));
    }

    @PutMapping("/{id}")
    public GoodsDto updateGoods(@PathVariable Long id, @RequestBody GoodsDto goods) {
        GoodsDto updatedOrder = goodsService.getGoods(id);
        updatedOrder.setName(goods.getName());
        updatedOrder.setPrice(goods.getPrice());
        return updatedOrder;
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<HttpStatus> deleteGoods(@PathVariable Long id) {
        try {
            goodsService.deleteGoods(id);
            return new ResponseEntity<>(HttpStatus.OK);
        } catch (ResourceNotFoundException e) {
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
    }
}
