import { Injectable } from '@angular/core';
import {
  HttpClient,
  HttpErrorResponse,
  HttpHeaders,
} from '@angular/common/http';
import { catchError, EMPTY, Observable } from 'rxjs';
import { environment } from '../../../environments/environment';
import { Order } from '../model/order.model';
import { PagedResponseModel } from '../model/paged-response.model';

@Injectable({
  providedIn: 'root',
})
export class OrderService {
  private readonly apiUrl = `${environment.API_ROOT}${environment.API_PREFIX}${environment.ORDER_URL}`;
  private page = 0;
  private size = 10;

  constructor(private http: HttpClient) {}

  getOrders(
    page = this.page,
    size = this.size
  ): Observable<PagedResponseModel<Order>> {
    const url = `${this.apiUrl}`;

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
      'X-Page-Current': String(page),
      'X-Page-Size': String(size),
    });
    return this.http.get<PagedResponseModel<Order>>(url, { headers });
  }

  getOrder(id: number): Observable<Order> {
    const url = `${this.apiUrl}/${id}`;
    return this.http.get<Order>(url);
  }

  putOrder(order: Order) {
    const url = `${this.apiUrl}/${order.id}`;
    return this.http.put<Order>(url, order);
  }

  postOrder(order: Order) {
    const url = this.apiUrl;
    return this.http.post<Order>(url, order);
  }

  deleteOrder(id: number) {
    const url = `${this.apiUrl}/${id}`;
    return this.http.delete(url).pipe(
      catchError((err: HttpErrorResponse) => {
        console.dir(err);
        return EMPTY;
      })
    );
  }
}
