import { Component, EventEmitter, Input, Output } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { Router } from '@angular/router';
import { environment } from '../../../../environments/environment';
import { Order } from '../../../core/model/order.model';
import { ControlsValidations } from '../../../core/model/validation.model';
import { OrderService } from '../../../core/service/order.service';

const DEV = !environment.production;

@Component({
  selector: 'app-order-form',
  templateUrl: './order-form.component.html',
  styleUrl: './order-form.component.scss',
})
export class OrderFormComponent {
  @Input() order!: Order;
  @Output() orderChange = new EventEmitter<Order>();

  form: FormGroup;
  validationMsgs: ControlsValidations;

  constructor(
    fb: FormBuilder,
    private router: Router,
    private orderService: OrderService,
  ) {
    this.form = fb.group({
      client: ['', Validators.required],
      address: ['', Validators.required],
      date: ['', Validators.required],
    });

    this.validationMsgs = {
      client: [{ type: 'required', message: 'Required fileld' }],
      address: [{ type: 'required', message: 'Required fileld' }],
      date: [
        { type: 'required', message: 'Required fileld' },
        { type: 'ngbDate', message: 'Invalid date format' },
      ],
    };
  }

  public get isNew(): boolean {
    return !this.order;
  }

  save() {
    if (this.form.valid && this.form.touched) {
      if (this.isNew) {
        this.orderService.postOrder({ ...this.form.value }).subscribe({
          next: (res) => {
            this.orderChange.emit(res);
            this.router.navigate(['order', res.id]);
          },
          error: (err) => {
            if (DEV) {
              console.dir(err);
            }
          },
        });
      } else {
        this.orderService
          .putOrder({ ...this.form.value, id: this.order.id })
          .subscribe({
            next: (res) => {
              this.orderChange.emit(res);
            },
            error: (err) => {
              if (DEV) {
                console.dir(err);
              }
            },
          });
      }
    }
  }

  ngOnInit(): void {
    if (!this.isNew) {
      this.form.setValue({
        client: this.order.client,
        date: this.order.date,
        address: this.order.address,
      });
    }
  }
}
