package com.example.testj.web.rest;

import com.example.testj.service.mapper.GoodsMapper;
import lombok.RequiredArgsConstructor;
import org.springframework.data.web.PagedModel;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.GoodsService;
import com.example.testj.service.dto.GoodsDto;

@RestController
@RequestMapping("/api/goods")
@CrossOrigin(origins = "http://localhost:4200")
@RequiredArgsConstructor
public class GoodsController {
    private final GoodsService goodsService;
    private final GoodsMapper goodsMapper;

    @GetMapping
    public PagedModel<GoodsDto> goods(
            @RequestHeader(name = "X-Page-Current", defaultValue = "0") Integer page,
            @RequestHeader(name = "X-Page-Size", defaultValue = "10") Integer size,
            @RequestHeader(name = "X-Sort-By", defaultValue = "id") String sortBy,
            @RequestHeader(name = "X-Sort-Direction", defaultValue = "ASC") String sortDir) {
        return goodsService.getGoodsPaged(page, size, sortBy, sortDir);
    }

    @GetMapping("/{id}")
    public GoodsDto goods(@PathVariable Long id) {
        return goodsService.getGoods(id);
    }

    @PostMapping
    @ResponseStatus(HttpStatus.CREATED)
    public GoodsDto createGoods(@RequestBody GoodsDto goods) {
        return this.goodsService.createOrUpdate(goodsMapper.toEntity(goods));
    }

    @PutMapping("/{id}")
    public GoodsDto updateGoods(@PathVariable Long id, @RequestBody GoodsDto goods) {
        GoodsDto updatedGoods = goodsService.getGoods(id);
        updatedGoods.setName(goods.getName());
        updatedGoods.setPrice(goods.getPrice());
        return this.goodsService.createOrUpdate(goodsMapper.toEntity(updatedGoods));
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<HttpStatus> deleteGoods(@PathVariable Long id) {
        try {
            goodsService.deleteGoods(id);
            return new ResponseEntity<>(HttpStatus.OK);
        } catch (ResourceNotFoundException e) {
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
    }
}
