package com.example.testj.service;

import com.example.testj.domain.Order;
import com.example.testj.repository.OrderRepository;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.impl.OrderServiceImpl;
import com.example.testj.service.mapper.OrderMapper;
import jakarta.validation.ConstraintViolationException;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mapstruct.factory.Mappers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PagedModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class OrderServiceTest {
    @InjectMocks
    private OrderServiceImpl orderService;

    @Mock
    private OrderRepository orderRepository;

    @Spy
    private OrderMapper orderMapper = Mappers.getMapper(OrderMapper.class);

    private static List<Order> ordersList = new ArrayList<>();

    @BeforeAll
    static void init() {
        Order order = new Order();
        order.setId(1L);
        order.setClient("client 1");
        order.setAddress("address 1");
        order.setDate("20-20-2020");

        Order order1 = new Order();
        order1.setId(2L);
        order1.setClient("client 2");
        order1.setAddress("address 2");
        order1.setDate("10-10-2021");

        ordersList = Stream.of(
                order,
                order1
        ).toList();
    }

    @Test
    void testCreateOrUpdateOrder() {
        // given
        Order order = new Order(null, "client new", "01-01-2021", "address new", null);

        Order result = new Order(
                ordersList.getLast().getId() + 1,
                order.getClient(),
                order.getDate(),
                order.getAddress(),
                null
        );
        // when
        when(orderRepository.save(order)).thenReturn(result);
        OrderDto created = orderService.createOrUpdateOrder(order);
        // then
        assertEquals(result.getId(), created.getId());
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    @Disabled
    void testSaveOrderWithEmptyClient() {
        // given
        Order order = new Order(null, "", "01-01-2020", "address new", null);
        // when
        assertThrows(ConstraintViolationException.class, () -> orderService.createOrUpdateOrder(order));
        // then
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    @Disabled
    void testSaveOrderWithNullName() {
        // given
        Order order = new Order(null, null,  "01-01-2020", "address", null);
        // when
        assertThrows(ConstraintViolationException.class, () -> orderService.createOrUpdateOrder(order));
        // then
        verify(orderRepository, times(1)).save(order);
    }

    @Test
    void testGetAllOrdersPaged() {
        // when
        when(orderRepository.findAll(any(Pageable.class))).thenReturn(new PageImpl<>(ordersList));
        PagedModel<OrderDto> ordersDtoPagedModel = orderService.getPagedOrders(1,10, "id", "ASC");
        // then
        assertNotNull(ordersDtoPagedModel);
        assertEquals(ordersDtoPagedModel.getContent().size(), 2);
        verify(orderRepository, times(1)).findAll(any(Pageable.class));
    }

    @Test
    void testGetOrder() {
        // given
        Long id = 1L;
        // when
        when(orderRepository.findById(id)).thenReturn(Optional.of(ordersList.getFirst()));
        OrderDto order = orderService.getOrder(id);
        // then
        assertNotNull(order);
        assertEquals(id, order.getId());
        verify(orderRepository, times(1)).findById(any());
    }

    @Test
    void testUpdateOrder() {
        // given
        Long id = 1L;
        Order order = ordersList.getFirst();
        order.setClient("client new");
        order.setAddress("address new");
        order.setDate("02-02-2020");
        // when
        when(orderRepository.save(order)).thenReturn(order);
        OrderDto updated = orderService.createOrUpdateOrder(order);
        // then
        assertNotNull(updated);
        assertEquals(id, updated.getId());
        assertEquals(ordersList.getFirst().getClient(), updated.getClient());
        assertEquals(ordersList.getFirst().getAddress(), updated.getAddress());
        verify(orderRepository, only()).save(any());
    }

    @Test
    void testDeleteOrder() {
        // given
        Long id = 2L;
        // when
        when(orderRepository.findById(id)).thenReturn(Optional.of(ordersList.getFirst()));
        doNothing().when(orderRepository).deleteById(id);
        orderService.deleteOrder(id);
        // then
        verify(orderRepository, times(1)).deleteById(id);
    }
}
