package com.example.testj.web.rest;

import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.GoodsService;
import com.example.testj.service.dto.GoodsDto;
import com.example.testj.service.mapper.GoodsMapper;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONObject;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.web.PagedModel;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;

import java.util.List;
import java.util.stream.Stream;

import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
public class GoodsControllerTest {

    @Autowired
    private MockMvc mvc;

    @MockBean
    private GoodsService goodsService;

    @MockBean
    private GoodsMapper goodsMapper;

    private final List<GoodsDto> goodsList = Stream.of(
            new GoodsDto(1L, "product 1", 9.99f),
            new GoodsDto(2L, "product 2", 25.9f),
            new GoodsDto(3L, "product 3", 5.12f),
            new GoodsDto(4L, "product 4", 55.29f),
            new GoodsDto(5L, "product 5", 0.99f)
    ).toList();

    @Test
    @DisplayName("GET /api/goods/:id [200 OK]")
    void testGetGoods() throws Exception {
        Long id = 1L;
        when(goodsService.getGoods(id)).thenReturn(goodsList.getFirst());
        mvc.perform(get("/api/goods/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.id").value(id))
                .andExpect(jsonPath("$.name").value(goodsList.getFirst().getName()));
    }

    @Test
    @DisplayName("GET /api/goods/:id [404 NotFound]")
    void testGetGoodNotFound() throws Exception {
        Long id = 999L;
        when(goodsService.getGoods(id)).thenThrow(new ResourceNotFoundException());
        mvc.perform(
                get("/api/goods/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isNotFound());
    }

    @Test
    @DisplayName("GET /api/goods [200 OK]")
    void testGetAllGoodsPaged() throws Exception {
        Page<GoodsDto> goodsPaged = new PageImpl<>(goodsList, PageRequest.of(0, 10),
                goodsList.size());
        when(goodsService.getPagedGoods(0, 10, "id", "ASC"))
                .thenReturn(new PagedModel<>(goodsPaged));
        mvc.perform(get("/api/goods").contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.content[0].id").value(1))
                .andExpect(jsonPath("$.page.totalElements").value(goodsList.size()));
    }

    @Test
    @DisplayName("POST /api/goods [201 Created]")
    void testCreateGoods() throws Exception {
        GoodsDto goodsDto = new GoodsDto(
                null,
                "product new",
                199.99
        );
        String body = new ObjectMapper().writeValueAsString(goodsDto);

        Long nextId = goodsList.getLast().getId() + 1;
        goodsDto.setId(nextId);
        when(goodsService.createOrUpdateGoods(goodsMapper.toEntity(goodsDto))).thenReturn(goodsDto);

        mvc.perform(post("/api/goods").contentType(MediaType.APPLICATION_JSON).content(body))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.id").isNumber())
                .andExpect(jsonPath("$.name").value(goodsDto.getName()))
                .andExpect(jsonPath("$.price").value(goodsDto.getPrice()));
    }

    @Test
    @DisplayName("POST /api/goods [400 Bad Request]")
    void testCreateGoodsBadRequest() throws Exception {
        JSONObject bodyJson = new JSONObject("{'name': null, 'price': 0}");

        mvc.perform(post("/api/goods").contentType(MediaType.APPLICATION_JSON).content(bodyJson.toString()))
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("PUT /api/goods/:id [200 OK]")
    void testUpdateGoods() throws Exception {
        Long id = 1L;
        GoodsDto goods = new GoodsDto();
        goods.setName("product name");
        goods.setPrice(0.99f);
        String body = new ObjectMapper().writeValueAsString(goods);

        goods.setId(id);
        when(goodsService.getGoods(id)).thenReturn(goodsList.getFirst());
        when(goodsService.createOrUpdateGoods(goodsMapper.toEntity(goods))).thenReturn(goods);

        mvc.perform(put("/api/goods/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(body))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.name").value(goods.getName()));
    }

    @Test
    @DisplayName("DELETE /api/goods/:id [200 OK]")
    void testDeleteGoods() throws Exception {
        Long id = 1L;
        doNothing().when(goodsService).deleteGoods(id);
        mvc.perform(delete("/api/goods/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk());
    }

    @Test
    @DisplayName("DELETE /api/goods/:id [404 Not Found]")
    void testDeleteGoodsNotFound() throws Exception {
        Long id = 999L;
        doThrow(new ResourceNotFoundException()).doNothing().when(goodsService).deleteGoods(id);
        mvc.perform(delete("/api/goods/{id}", id))
                .andExpect(status().isNotFound());
    }
}
