package com.example.testj.service;

import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PagedModel;
import org.springframework.stereotype.Service;

import com.example.testj.domain.Order;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.repository.OrderRepository;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.utils.MappingUtils;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;


@Service
public class OrderService {
    private final OrderRepository orderRepository;
    private final MappingUtils mappingUtils;

    public OrderService(OrderRepository orderRepository, MappingUtils mappingUtils) {
        this.orderRepository = orderRepository;
        this.mappingUtils = mappingUtils;
    }

    public PagedModel<OrderDto> getOrders(Integer page, Integer size, String sortBy, String sortDir) {
        return new PagedModel<>(orderRepository
                .findAll(PageRequest.of(page, size, Sort.by(Sort.Direction.fromString(sortDir), sortBy)))
                .map(mappingUtils::toOrderDto));
    }

    public OrderDto getOrder(Long id) throws ResourceNotFoundException {
        return mappingUtils.toOrderDto(orderRepository.findById(id).orElseThrow(ResourceNotFoundException::new));
    }

    public void deleteOrder(Long id) throws ResourceNotFoundException {
        orderRepository.findById(id).orElseThrow(ResourceNotFoundException::new);
        orderRepository.deleteById(id);
    }

    public OrderDto createOrUpdateOrder(Order order) {
        return mappingUtils.toOrderDto(orderRepository.save(order));
    }

    public OrderDto patchOrder(Long id, JsonNode patch) throws JsonMappingException {
        Order orderData = orderRepository.findById(id).orElseThrow(ResourceNotFoundException::new);
        Order patchedOrder = new ObjectMapper().updateValue(orderData, patch);
        return mappingUtils.toOrderDto(orderRepository.save(patchedOrder));
    }
}
