import { Component, OnInit } from '@angular/core';
import { NgbModal } from '@ng-bootstrap/ng-bootstrap';
import { Subject, concatMap, filter, take } from 'rxjs';
import { AbstractBasicComponent } from '../../core/abstract/abstract-component';
import { Goods } from '../../core/model/goods.model';
import { Order } from '../../core/model/order.model';
import {
  PageModel,
  PagedResponseModel,
} from '../../core/model/paged-response.model';
import { GoodsService } from '../../core/service/goods.service';
import { OrderService } from '../../core/service/order.service';
import { GoodsEditModalComponent } from './components/goods-edit-modal/goods-edit-modal.component';
import { OrderEditModalComponent } from './components/order-edit-modal/order-edit-modal.component';

@Component({
  selector: 'app-index-page',
  templateUrl: './index-page.component.html',
  styleUrl: './index-page.component.scss',
})
export class IndexPageComponent
  extends AbstractBasicComponent
  implements OnInit
{
  protected ordersSubj$ = new Subject<PagedResponseModel<Order>>();
  protected goodsSubj$ = new Subject<PagedResponseModel<Goods>>();

  goodsPage!: PageModel;
  orderPage!: PageModel;

  constructor(
    private orderService: OrderService,
    private goodsService: GoodsService,
    private modalService: NgbModal,
  ) {
    super();
  }

  ngOnInit() {
    this.orderService.getOrders().subscribe((resp) => {
      this.orderPage = resp.page;
      this.ordersSubj$.next(resp);
    });

    this.goodsService.get().subscribe((resp) => {
      this.goodsPage = resp.page;
      this.goodsSubj$.next(resp);
    });
  }

  ordersPageNumber(page: number) {
    this.orderService.getOrders(page).subscribe((resp) => {
      this.orderPage = resp.page;
      this.ordersSubj$.next(resp);
    });
  }

  goodsPageNumber(page: number) {
    this.goodsService.get(page).subscribe((resp) => {
      this.goodsPage = resp.page;
      this.goodsSubj$.next(resp);
    });
  }

  deleteOrder(id: number) {
    this.orderService
      .deleteOrder(id)
      .pipe(concatMap(() => this.orderService.getOrders(this.orderPage.number)))
      .subscribe((resp) => {
        this.orderPage = resp.page;
        this.ordersSubj$.next(resp);
      });
  }

  deleteGoods(id: number) {
    this.goodsService
      .delete(id)
      .pipe(concatMap(() => this.goodsService.get(this.goodsPage.number)))
      .subscribe((resp) => {
        this.goodsPage = resp.page;
        this.goodsSubj$.next(resp);
      });
  }

  openOrderModal(order: Order): void {
    const orderModalRef = this.modalService.open(OrderEditModalComponent);
    orderModalRef.componentInstance.order = order;
    orderModalRef.componentInstance.orderChange
      .pipe(
        take(1),
        concatMap(() => this.orderService.getOrders(this.orderPage.number)),
      )
      .subscribe((resp: PagedResponseModel<Order>) => {
        this.orderPage = resp.page;
        this.ordersSubj$.next(resp);
      });
  }

  openGoodsModal(product: Goods): void {
    const goodsModalRef = this.modalService.open(GoodsEditModalComponent);
    goodsModalRef.componentInstance.product = product;
    goodsModalRef.componentInstance.isUpdate
      .pipe(
        filter(Boolean),
        concatMap(() => this.goodsService.get(this.goodsPage.number)),
      )
      .subscribe((resp: PagedResponseModel<Goods>) => {
        this.goodsPage = resp.page;
        this.goodsSubj$.next(resp);
      });
  }

  openNewProductModal(): void {
    const goodsModalRef = this.modalService.open(GoodsEditModalComponent);
    goodsModalRef.componentInstance.isUpdate
      .pipe(
        filter(Boolean),
        concatMap(() => this.goodsService.get(this.goodsPage.number)),
      )
      .subscribe((resp: PagedResponseModel<Goods>) => {
        console.dir(resp);
        this.goodsPage = resp.page;
        this.goodsSubj$.next(resp);
      });
  }
}
