package com.example.testj.service;

import com.example.testj.domain.Goods;
import com.example.testj.repository.GoodsRepository;
import com.example.testj.service.dto.GoodsDto;
import com.example.testj.service.impl.GoodsServiceImpl;
import com.example.testj.service.mapper.GoodsMapper;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mapstruct.factory.Mappers;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PagedModel;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class GoodsServiceTest {
    @InjectMocks
    private GoodsServiceImpl goodsService;

    @Mock
    private GoodsRepository goodsRepository;

    @Spy
    private final GoodsMapper goodsMapper = Mappers.getMapper(GoodsMapper.class);

    private final List<Goods> goodsList = Stream.of(
            new GoodsDto(1L, "product #1", 99.99f),
            new GoodsDto(2L, "product #2", 25.99f),
            new GoodsDto(3L, "product #3", 3.5f),
            new GoodsDto(4L, "product #4", 60f),
            new GoodsDto(5L, "product #5", 10.1f)
    ).map(goodsMapper::toEntity).toList();

    @Test
    void testGetGoods() {
        // given
        Long id = 1L;
        // when
        when(goodsRepository.findById(id)).thenReturn(Optional.of(goodsList.getFirst()));
        GoodsDto goodsDto = goodsService.getGoods(id);
        // then
        assertEquals(id, goodsDto.getId());
        verify(goodsRepository, times(1)).findById(id);
    }

    @Test
    void testGetPagedGoods() {
        // when
        when(goodsRepository.findAll(any(Pageable.class))).thenReturn(new PageImpl<>(goodsList));
        PagedModel<GoodsDto> goodsDtoPagedModel = goodsService.getPagedGoods(1,10, "id", "ASC");
        // then
        assertNotNull(goodsDtoPagedModel);
        assertEquals(goodsDtoPagedModel.getContent().size(), 5);
        verify(goodsRepository, times(1)).findAll(any(Pageable.class));
    }

    @Test
    void testUpdateGoods() {
        // given
        Goods goods = goodsList.getFirst();
        goods.setName("updated #1");
        goods.setPrice(1.5f);
        // when
        when(goodsRepository.save(goods)).thenReturn(goods);
        GoodsDto updated = goodsService.createOrUpdateGoods(goods);
        // then
        assertNotNull(updated);
        assertEquals(goods.getId(), updated.getId());
        assertEquals(goods.getPrice(), updated.getPrice());
        assertEquals(goods.getName(), updated.getName());
        verify(goodsRepository, only()).save(any());
   }

   @Test
   void testCreateGoods() {
       // given
       Goods goods = new Goods();
       goods.setName("New Product");
       goods.setPrice(99.99f);
       Goods result = new Goods();
       Long goodsNextId = goodsList.getLast().getId() + 1;
       result.setId(goodsNextId);
       result.setName(goods.getName());
       result.setPrice(goods.getPrice());
       // when
       when(goodsRepository.save(goods)).thenReturn(result);
       GoodsDto created = goodsService.createOrUpdateGoods(goods);
       // then
       assertNotNull(goods);
       assertEquals(goodsNextId, created.getId());
       assertEquals(created.getName(), goods.getName());
       assertEquals(created.getPrice(), 99.99f);
       verify(goodsRepository, only()).save(any());
   }

   @Test
   void testDeleteGoods() {
       // given
       Long id = 1L;
       // when
       when(goodsRepository.findById(id)).thenReturn(Optional.of(goodsList.getFirst()));
       doNothing().when(goodsRepository).deleteById(id);
       goodsService.deleteGoods(id);
       // then
       verify(goodsRepository, times(1)).deleteById(id);
   }
}
