package com.example.testj.web.rest;

import com.example.testj.service.OrderService;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.mapper.OrderMapper;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.web.PagedModel;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
//@WebMvcTest(OrderController.class)
public class OrderControllerTest {
    @Autowired
    private MockMvc mvc;

    @MockBean
    private OrderService orderService;

    @MockBean
    private OrderMapper orderMapper;

    private static List<OrderDto> ordersList = new ArrayList<>();

    @BeforeAll
    static void init() {
        OrderDto orderDto = new OrderDto();
        orderDto.setId(1L);
        orderDto.setClient("client 1");
        orderDto.setAddress("address 1");
        orderDto.setDate("20-20-2020");

        OrderDto orderDto1 = new OrderDto();
        orderDto1.setId(2L);
        orderDto1.setClient("client 2");
        orderDto1.setAddress("address 2");
        orderDto1.setDate("10-10-2021");

        ordersList = Stream.of(
                orderDto,
                orderDto1
        ).toList();
    }

    @Test
    @DisplayName("GET /api/orders")
    public void orders() throws Exception {
        Page<OrderDto> ordersPaged = new PageImpl<>(ordersList, PageRequest.of(0, 10),
                ordersList.size());
        when(orderService.getPagedOrders(0, 10, "id", "ASC"))
                .thenReturn(new PagedModel<>(ordersPaged));
        mvc.perform(get("/api/orders").contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.content[0].id").value(1))
                .andExpect(jsonPath("$.page.totalElements").value(ordersList.size()));
    }

    @Test
    @DisplayName("GET /api/orders/:id")
    public void order() throws Exception {
        Long id = 1L;
        when(orderService.getOrder(id)).thenReturn(ordersList.getFirst());
        mvc.perform(get("/api/orders/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.id").value(id))
                .andExpect(jsonPath("$.client").value(ordersList.getFirst().getClient()))
                .andExpect(jsonPath("$.address").value(ordersList.getFirst().getAddress()));
    }

    @Test
    @DisplayName("POST /api/orders [201 Created]")
    void testCreateOrder() throws Exception {
        OrderDto orderDto = new OrderDto();
        orderDto.setClient("new client");
        orderDto.setAddress("new address");
        orderDto.setDate("20-20-2020");
        String body = new ObjectMapper().writeValueAsString(orderDto);

        Long nextId = (ordersList.getLast().getId() + 1);
        orderDto.setId(nextId);

        when(orderService.createOrUpdateOrder(orderMapper.toEntity(orderDto))).thenReturn(orderDto);

        mvc.perform(post("/api/orders").contentType(MediaType.APPLICATION_JSON).content(body))
                .andExpect(status().isCreated());
//                .andExpect(jsonPath("$.id").isNumber())
//                .andExpect(jsonPath("$.client").value(orderDto.getClient()))
//                .andExpect(jsonPath("$.address").value(orderDto.getAddress()))
//                .andExpect(jsonPath("$.date").value(orderDto.getDate()));
    }

    @Test
    @DisplayName("PUT /api/orders/:id")
    void testUpdateOrder() throws Exception {
        Long id = 1L;
        OrderDto orderDto = new OrderDto();
        orderDto.setClient("new client");
        orderDto.setAddress("new address");
        orderDto.setDate("21-21-2021");
        String body = new ObjectMapper().writeValueAsString(orderDto);

        orderDto.setId(id);
        when(orderService.getOrder(id)).thenReturn(ordersList.getFirst());
        when(orderService.createOrUpdateOrder(orderMapper.toEntity(orderDto))).thenReturn(orderDto);

        mvc.perform(put("/api/orders/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(body))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.client").value(orderDto.getClient()))
                .andExpect(jsonPath("$.address").value(orderDto.getAddress()))
                .andExpect(jsonPath("$.date").value(orderDto.getDate()));
    }

    @Test
    @DisplayName("DELETE /api/orders/:id [200 OK]")
    void testDeleteOrder() throws Exception {
        Long id = 1L;
        doNothing().when(orderService).deleteOrder(id);
        mvc.perform(delete("/api/orders/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk());
    }
}
