package com.example.testj.web.rest;

import com.example.testj.domain.Goods;
import com.example.testj.domain.Order;
import com.example.testj.domain.OrderLine;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.OrderLineService;
import com.example.testj.service.dto.GoodsDto;
import com.example.testj.service.dto.OrderLineDto;
import com.example.testj.service.dto.OrderLineRequestDto;
import com.example.testj.service.mapper.OrderLineMapper;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;

import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
//@WebMvcTest(OrderLineController.class)
public class OrderLineControllerTest {
    @Autowired
    private MockMvc mvc;

    @MockBean
    private OrderLineService orderLineService;

    @MockBean
    private OrderLineMapper orderLineMapper;

    @Test
    @DisplayName("GET /api/order-line/:id [200 OK]")
    void testGetOrderLine() throws Exception {
        Long id = 1L;
        GoodsDto goodsDto = new GoodsDto(1L, "product 1", 100.00f);
        OrderLineDto orderLineDto = new OrderLineDto();
        orderLineDto.setId(id);
        orderLineDto.setCount(10);
        orderLineDto.setGoods(goodsDto);
        when(orderLineService.getOrderLine(id)).thenReturn(orderLineDto);
        mvc.perform(get("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.id").value(id))
                .andExpect(jsonPath("$.count").value(orderLineDto.getCount()))
                .andExpect(jsonPath("$.goods.id").value(goodsDto.getId()));
    }

    @Test
    @DisplayName("GET /api/order-line/:id [404 Not Found]")
    void testGetOrderLineNotFound() throws Exception {
        Long id = 999L;
        when(orderLineService.getOrderLine(id)).thenThrow(new ResourceNotFoundException());
        mvc.perform(get("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isNotFound());
    }

    @Test
    @DisplayName("POST /api/order-line [201 Created]")
    void testCreateOrderLine() throws Exception {
        Goods goods = new Goods();
        goods.setId(1L);
        goods.setName("product 1");
        goods.setPrice(100.00);
        Order order = new Order();
        order.setId(1L);

        OrderLine orderLine = new OrderLine();
        orderLine.setOrder(order);
        orderLine.setGoods(goods);
        orderLine.setCount(10);

        OrderLineRequestDto requestDto = new OrderLineRequestDto();
        requestDto.setGoodsId(orderLine.getGoods().getId());
        requestDto.setOrderId(orderLine.getOrder().getId());
        requestDto.setCount(orderLine.getCount());

        String body = new ObjectMapper().writeValueAsString(requestDto);

        Long nextId = 2L;
        orderLine.setId(nextId);
        OrderLineDto orderLineDto = orderLineMapper.toDto(orderLine);
//        OrderLineDto orderLineDto = new OrderLineDto(nextId, new GoodsDto(1L, "product 1", 100.00), 10);
        when(orderLineService.createOrderLine(requestDto)).thenReturn(orderLineDto);


        mvc.perform(post("/api/order-line").contentType(MediaType.APPLICATION_JSON).content(body))
                .andExpect(status().isCreated())
                .andReturn();
//                .andExpectAll(
//                        content().contentType(MediaType. APPLICATION_JSON),
//                        jsonPath("$.id").value(orderLineDto.getId()),
//                        jsonPath("$.count").value(orderLineDto.getCount()));
    }

    @Test
    @DisplayName("PUT /api/order-line/:id [200 OK]")
    void testUpdateOrderLine() throws Exception {
        Long id = 1L;
        OrderLineDto orderLineDto = new OrderLineDto();
        orderLineDto.setId(id);
        orderLineDto.setCount(100);

        OrderLineRequestDto orderLineRequest = new OrderLineRequestDto();
        orderLineRequest.setCount(orderLineDto.getCount());
        String body = new ObjectMapper().writeValueAsString(orderLineRequest);

        when(orderLineService.updateOrderLine(orderLineMapper.toEntity(orderLineDto))).thenReturn(orderLineDto);

        mvc.perform(put("/api/order-line/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(body))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.count").value(orderLineDto.getCount()));
    }

    @Test
    @DisplayName("DELETE /api/order-line/:id [200 OK]")
    void testDeleteOrderLine() throws Exception {
        Long id = 1L;
        doNothing().when(orderLineService).deleteOrderLine(id);
        mvc.perform(delete("/api/order-line/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk());
    }
}
