package com.example.testj.web.rest;

import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.OrderService;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.mapper.OrderMapper;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.hamcrest.Matchers;
import org.json.JSONObject;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.web.PagedModel;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
public class OrderControllerTest {
    @Autowired
    private MockMvc mvc;

    @Autowired
    private ObjectMapper objectMapper;

    @MockBean
    private OrderService orderService;

    @MockBean
    private OrderMapper orderMapper;

    private static List<OrderDto> ordersList = new ArrayList<>();

    @BeforeAll
    static void init() {
        ordersList = Stream.of(
                new OrderDto(
                        1L, "client 1", LocalDate.of(2020, 1, 20), "address 1", null
                ),
                new OrderDto(
                        2L, "client 2", LocalDate.of(2020, 1, 21), "address 2", null
                )
        ).toList();
    }

    @Test
    @DisplayName("GET /api/orders")
    public void orders() throws Exception {
        Page<OrderDto> ordersPaged = new PageImpl<>(ordersList, PageRequest.of(0, 10),
                ordersList.size());
        when(orderService.getPagedOrders(0, 10, "id", "ASC"))
                .thenReturn(new PagedModel<>(ordersPaged));
        mvc.perform(get("/api/orders").contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.content[0].id").value(1))
                .andExpect(jsonPath("$.page.totalElements").value(ordersList.size()));
    }

    @Test
    @DisplayName("GET /api/orders/:id")
    public void order() throws Exception {
        Long id = 1L;
        when(orderService.getOrder(id)).thenReturn(ordersList.getFirst());
        mvc.perform(get("/api/orders/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.id").value(id))
                .andExpect(jsonPath("$.client").value(ordersList.getFirst().getClient()))
                .andExpect(jsonPath("$.address").value(ordersList.getFirst().getAddress()));
    }

    @Test
    @DisplayName("POST /api/orders [201 Created]")
    void testCreateOrder() throws Exception {
        OrderDto orderDto = new OrderDto(
               null, "client new", LocalDate.of(2021, 1, 21), "address new", null
        );
        String body = objectMapper.writeValueAsString(orderDto);
        Long nextId = (ordersList.getLast().getId() + 1);
        orderDto.setId(nextId);

        when(orderService.createOrUpdateOrder(orderMapper.toEntity(orderDto))).thenReturn(orderDto);

        mvc.perform(post("/api/orders").contentType(MediaType.APPLICATION_JSON).content(body))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.id").isNumber())
                .andExpect(jsonPath("$.client").value(orderDto.getClient()))
                .andExpect(jsonPath("$.address").value(orderDto.getAddress()))
                .andExpect(jsonPath("$.date").value(orderDto.getDate().toString())).andReturn();
    }

    @Test
    @DisplayName("POST /api/orders [400 Bad Request]")
    void testCreateOrderBadRequest() throws Exception {
        JSONObject bodyJson = new JSONObject("{'client': null, 'address': '', 'date': ''}");

        mvc.perform(post("/api/orders").contentType(MediaType.APPLICATION_JSON).content(bodyJson.toString()))
                .andExpect(status().isBadRequest())
                .andExpect(jsonPath("$.errors[*].field").value(
                        Matchers.containsInAnyOrder("client", "address", "date")));
    }

    @Test
    @DisplayName("PUT /api/orders/:id")
    void testUpdateOrder() throws Exception {
        Long id = 1L;
        OrderDto orderDto = new OrderDto(
                id, "client new", LocalDate.of(2021,1,21), "address new", null
        );
        String body = objectMapper.writeValueAsString(orderDto);

        when(orderService.getOrder(id)).thenReturn(ordersList.getFirst());
        when(orderService.createOrUpdateOrder(orderMapper.toEntity(orderDto))).thenReturn(orderDto);

        mvc.perform(put("/api/orders/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(body))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.client").value(orderDto.getClient()))
                .andExpect(jsonPath("$.address").value(orderDto.getAddress()))
                .andExpect(jsonPath("$.date").value(orderDto.getDate().toString()));
    }

    @Test
    @DisplayName("PATCH /api/orders/:id [200 OK]")
    void testPatchOrder() throws Exception {
        Long id = 1L;
        OrderDto orderDto = new OrderDto(
                id, "client patch", ordersList.getFirst().getDate(), ordersList.getFirst().getAddress(),
                ordersList.getFirst().getOrderLines());
        String patch = objectMapper.writeValueAsString("{'name': 'client patch'}");

        when(orderService.getOrder(id)).thenReturn(ordersList.getFirst());
        when(orderService.patchOrder(id, objectMapper.readTree(patch))).thenReturn(orderDto);

        mvc.perform(patch("/api/orders/{id}", id)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(patch))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.client").value(orderDto.getClient()))
                .andExpect(jsonPath("$.address").value(orderDto.getAddress()))
                .andExpect(jsonPath("$.date").value(orderDto.getDate().toString()));
    }

    @Test
    @DisplayName("DELETE /api/orders/:id [200 OK]")
    void testDeleteOrder() throws Exception {
        Long id = 1L;
        doNothing().when(orderService).deleteOrder(id);
        mvc.perform(delete("/api/orders/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk());
    }

    @Test
    @DisplayName("DELETE /api/orders/:id [404 Not Found]")
    void testDeleteOrderNotFound() throws Exception {
        Long id = 999L;
        doThrow(ResourceNotFoundException.class).doNothing().when(orderService).deleteOrder(id);
        mvc.perform(delete("/api/orders/{id}", id).contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isNotFound());
    }
}
