package com.example.testj.service.impl;

import com.example.testj.domain.Order;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.repository.OrderRepository;
import com.example.testj.service.OrderService;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.mapper.OrderMapper;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PagedModel;
import org.springframework.stereotype.Service;


@Service
@RequiredArgsConstructor
public class OrderServiceImpl implements OrderService {
    private final OrderRepository orderRepository;
    private final OrderMapper orderMapper;

    @Override
    public PagedModel<OrderDto> getPagedOrders(Integer page, Integer size, String sortBy, String sortDir) {
        return new PagedModel<>(orderRepository
                .findAll(PageRequest.of(page, size, Sort.by(Sort.Direction.fromString(sortDir), sortBy)))
                .map(orderMapper::toDto));
    }

    @Override
    public OrderDto getOrder(Long id) throws ResourceNotFoundException {
        return orderRepository.findById(id).map(orderMapper::toDto).orElseThrow(ResourceNotFoundException::new);
    }

    @Override
    public void deleteOrder(Long id) throws ResourceNotFoundException {
        orderRepository.findById(id).orElseThrow(ResourceNotFoundException::new);
        orderRepository.deleteById(id);
    }

    @Override
    public OrderDto createOrUpdateOrder(Order order) {
        return orderMapper.toDto(orderRepository.save(order));
    }

    @Override
    public OrderDto patchOrder(Long id, JsonNode patch) throws JsonMappingException {
        Order orderData = orderRepository.findById(id).orElseThrow(ResourceNotFoundException::new);
        Order patchedOrder = new ObjectMapper().updateValue(orderData, patch);
        return orderMapper.toDto(orderRepository.save(patchedOrder));
    }
}
