package com.example.testj.web.rest;

import com.example.testj.domain.Order;
import com.example.testj.exception.ResourceNotFoundException;
import com.example.testj.service.OrderService;
import com.example.testj.service.dto.OrderDto;
import com.example.testj.service.mapper.OrderMapper;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import jakarta.validation.Valid;
import lombok.RequiredArgsConstructor;
import org.springframework.data.web.PagedModel;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping("/api/orders")
@CrossOrigin(origins = "http://localhost:4200")
@RequiredArgsConstructor
public class OrderController {
    private final OrderService orderService;
    private final OrderMapper orderMapper;

    @GetMapping
    public PagedModel<OrderDto> getPaged(
            @RequestHeader(name = "X-Page-Current", defaultValue = "0") Integer page,
            @RequestHeader(name = "X-Page-Size", defaultValue = "10") Integer size,
            @RequestHeader(name = "X-Sort-By", defaultValue = "id") String sortBy,
            @RequestHeader(name = "X-Sort-Direction", defaultValue = "ASC") String sortDir) {
        return orderService.getPagedOrders(page, size, sortBy, sortDir);
    }

    @GetMapping("/{id}")
    public OrderDto get(@PathVariable Long id) {
        return orderService.getOrder(id);
    }

    @PostMapping
    @ResponseStatus(HttpStatus.CREATED)
    public OrderDto create(@Valid @RequestBody Order order) {
        return orderService.createOrUpdateOrder(order);
    }

    @PutMapping("/{id}")
    @ResponseStatus(HttpStatus.OK)
    public OrderDto update(@PathVariable Long id, @RequestBody OrderDto order) {

        OrderDto updatedOrder = orderService.getOrder(id);
        updatedOrder.setClient(order.getClient());
        updatedOrder.setAddress(order.getAddress());
        updatedOrder.setDate(order.getDate());

        return orderService.createOrUpdateOrder(orderMapper.toEntity(updatedOrder));
    }

    @PatchMapping("/{id}")
    public OrderDto patch(@PathVariable Long id, @RequestBody JsonNode patch) throws JsonMappingException {
        return orderService.patchOrder(id, patch);
    }

    @DeleteMapping("/{id}")
    public ResponseEntity<HttpStatus> delete(@PathVariable Long id) throws ResourceNotFoundException {
        try {
            orderService.deleteOrder(id);
            return new ResponseEntity<>(HttpStatus.OK);
        } catch (ResourceNotFoundException e) {
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
    }
}
